package fractsplinewavelets;

import data.*;
import ij.*;

/**
 * <hr>
 * <p><b>Plugin of ImageJ:</b><br>
 * Fractional Spline Wavelet Module<br>
 *
 * <p><b>Authors:</b><br>
 * Daniel Sage, Dimitri Van De Ville
 * <a href="mailto:daniel.sage@epfl.ch?subject=Fractional Spline Wavelet Plugin">daniel.sage@epfl.ch</a><br>
 * Swiss Federal Institute of Technology Lausanne,
 * Biomedical Imaging Group,
 * CH-1015 Lausanne, Switzerland,
 * <a href="http://bigwww.epfl.ch">http://bigwww.epfl.ch</a><br>
 *
 * <p><b>Version:</b><br>
 * April 2003<br>
 *
 * <p><b>Copyright</b><br>
 * Copyright  2003, Swiss Federal Institute of Technology, Lausanne, Switzerland, (EPFL)<br>
 *
 * <hr>
 *
 * <p><b>Purpose of the class:</b><br>
 * Some example of 3D wavelet coefficients processing.
 *
 *
 * <hr>
 *
 * <p><b>How to compile your own processing function</b><br>
 * 1) Implement your algorithm in the function "DoProcessing3D" below. 
 * 2) Compile the modified java source file with a Java compiler, e.g.:
 *       javac -classpath ../data/data.jar:../../../ij.jar CoefProcessing.java
 *    The ImageAccess library (data.jar) can be found in the archive of this plug-in.
 *    You also need the jar-version (ij.jar) of ImageJ, downloadable from the ImageJ site.
 * 3) Start the plug-in, and select "Your own processing"  
 */

final public class CoefProcessing
{

/**
* Template for processing 3D wavelet coefficients.
*
* @param coef			a Data object representing the wavelet coefficients
* @param interations	the number of iterations along each dimension
*/
static final public void doProcessing3D(Data coef, final int[] iterations)
{
	// get the iterations along each dimension
	int iterationX=iterations[0];
	int iterationY=iterations[1];
	int iterationZ=iterations[2];

	// get the size along each dimension
	int nx = coef.getXSize();
	int ny = coef.getYSize();
	int nz = coef.getZSize();
	
	final int type = coef.getType();
	
	// we're going to get and put by block processing
	coef.setDirection(Data.XYZ);

	// auxilary matrix to facilitate the loop processing each 3D subband
    final boolean offsets[][] = {
      /* LLL */ { false, false, false },
      /* LLH */ { false, false, true  },
      /* LHL */ { false, true,  false },
      /* LHH */ { false, true,  true  },
      /* HLL */ { true,  false, false },
      /* HLH */ { true,  false, true  },
      /* HHL */ { true,  true,  false },
      /* HHH */ { true,  true,  true  }
    };

	// data block to process subband
	Data sub;

	// outer loop: continue processing up to maximal number of iterations	
	for ( int i=0; i<Math.max(iterationX,Math.max(iterationY,iterationZ)); i++) {

		// adapt dimensions of subband along a dimensions
		if (iterationX>i) nx = nx / 2;
		if (iterationY>i) ny = ny / 2;
		if (iterationZ>i) nz = nz / 2;
		
		// create empty block at size of subband
		sub = Data.create(nx, ny, nz, type);
		
		// process all subbands except LLL (2^3-1)
		for ( int j=1; j<8; j++ ) {
		
			// do only try to access a highpass along a direction if it is available
			if ( !( (iterationX<=i && offsets[j][0]) ||
				    (iterationY<=i && offsets[j][1]) ||
				    (iterationZ<=i && offsets[j][2]) ) ) 
			{
				// read the subband in to the data block
				coef.get( (offsets[j][0]) ? nx : 0,
						  (offsets[j][1]) ? ny : 0,
						  (offsets[j][2]) ? nz : 0 , sub);
						
				// ...  
				// ...your processing here... 
				// ...
				
				// write out the processed subband
				coef.put( (offsets[j][0]) ? nx : 0,
						  (offsets[j][1]) ? ny : 0,
						  (offsets[j][2]) ? nz : 0 , sub);
			}
		}
	}

	// finally process the remaining lowpass band
	sub = Data.create(nx,ny,nz,type);

	// ...
	// ...your processing here...
	// ...

	coef.put(0, 0, 0, sub);
}

/**
* Apply a soft thresholding to all coefficients.
*
* All the image values lower than '-threshold' and greater than 'threshold'
* are set to 0. The remaining positive values are reduced by 'threshold';
* the remaining negative values are augmented by 'threshold'.
*
* @param coef			a Data object representing the wavelet coefficients
* @param threshold      a value for the threshold
*/
static final public void doSoftThreshold3D(Data coef, final double threshold)
{
	final int nx = coef.getXSize();
	final int ny = coef.getYSize();
	final int nz = coef.getZSize();
	double pixel=0.0;
	
	double[] row = new double[nx];
	coef.setDirection(Data.X);
	
	for (int z=0; z<nz; z++)
  	  for (int y=0; y<ny; y++) {
		coef.get(0, y, z, row);
		for (int x=0; x<nx; x++) {
			if (row[x] < -threshold)
				row[x] = row[x] + threshold;
			else if (row[x] > threshold)
				row[x] = row[x] - threshold;
			else
				row[x] = 0.0;
		}
		coef.put(0, y, z, row);
	  }
}

/**
* Apply a hard thresholding to all coefficients.
*
* All the image values lower than '-threshold' and greater than 'threshold'
* are set to 0.
*
* @param coef			a Data representing the wavelet coefficients
* @param threshold      a value for the threshold
*/
static final public void doHardThreshold3D(Data coef, final double threshold)
{
	final int nx = coef.getXSize();
	final int ny = coef.getYSize();
	final int nz = coef.getZSize();
	
	double pixel=0.0;
	
	double[] row = new double[nx];
	coef.setDirection(Data.X);
	
	for (int z=0; z<nz; z++)
	  for (int y=0; y<ny; y++) {
		coef.get(0, y, z, row);
		for (int x=0; x<nx; x++) {
			if ((row[x] >= -threshold) && (row[x] <= threshold)) {
				row[x] = 0.0;
			}
		}
		coef.put(0, y, z, row);
	  }
}

/**
* Compute the number of non zero pixels.
*
* @param coef			a Data object representing the wavelet coefficients
* @param factor			the enhancement factor	
*/
static final public void doEnhancement3D(Data coef, final double factor, final int iterations[])
{
	final int nx0 = coef.getXSize()/(int)Math.pow((double)2,(double)(iterations[0]));
	final int ny0 = coef.getYSize()/(int)Math.pow((double)2,(double)(iterations[1]));
	final int nz0 = coef.getZSize()/(int)Math.pow((double)2,(double)(iterations[2]));
	
	Data sub = Data.create(nx0, ny0, nz0, coef.getType());
	Data coefmultiply = null;
	coef.get(0,0,0, sub);
	coefmultiply = Operator.multiply(coef, factor);
	coef.copy(coefmultiply);
	coef.put(0,0,sub);
}

/**
* Apply a rescaling.
*
* Stretches the contrast inside subbands
* so that the gray levels are in the range : the positivy values
* are stretched to center-max and the negative values are stretched 
* to min-center and a contrast stretching is applied on the low band
*
* @param coef			a Data object representing the wavelet coefficients
* @param iterations    	number of iterations
*/
static final public void doRescale3D(Data coef, final int[] iterations)
{
	int iterationX=iterations[0];
	int iterationY=iterations[1];
	int iterationZ=iterations[2];

	int nx = coef.getXSize();
	int ny = coef.getYSize();
	int nz = coef.getZSize();
	
	final int type = coef.getType();
	
	coef.setDirection(Data.XYZ);

    final boolean offsets[][] = {
      /* LLL */ { false, false, false },
      /* LLH */ { false, false, true  },
      /* LHL */ { false, true,  false },
      /* LHH */ { false, true,  true  },
      /* HLL */ { true,  false, false },
      /* HLH */ { true,  false, true  },
      /* HHL */ { true,  true,  false },
      /* HHH */ { true,  true,  true  }
    };

	Data sub;
	double subd[][][];
	
	for ( int i=0; i<Math.max(iterationX,Math.max(iterationY,iterationZ)); i++) {
		if (iterationX>i) nx = nx / 2;
		if (iterationY>i) ny = ny / 2;
		if (iterationZ>i) nz = nz / 2;
		
		sub = Data.create(nx, ny, nz, type);
		
		// all subbands except LLL
		for ( int j=1; j<8; j++ ) {
			if ( !( (iterationX<=i && offsets[j][0]) ||
				    (iterationY<=i && offsets[j][1]) ||
				    (iterationZ<=i && offsets[j][2]) ) ) {
				coef.get( (offsets[j][0]) ? nx : 0,
						  (offsets[j][1]) ? ny : 0,
						  (offsets[j][2]) ? nz : 0 , sub); 
				normalizeContrastCentered(sub);
				coef.put( (offsets[j][0]) ? nx : 0,
						  (offsets[j][1]) ? ny : 0,
						  (offsets[j][2]) ? nz : 0 , sub);
			}
		}
	}

	// lowest LLL
	sub = Data.create(nx,ny,nz,type);
	coef.get(0, 0, 0, sub);
	sub.normalizeContrast();
	coef.put(0, 0, 0, sub);
}

/**
* Stretches the contrast inside an image so that the gray levels are
* in the range : the positivy values are stretched to center-max 
* and the negative values are stretched to min-center.
*/
private static void normalizeContrastCentered(Data image)
{
	double maxGoal = 255.0;
	double minGoal = 0.0;
	double center = (maxGoal+minGoal)/2.0;
	
	// Get the min and max
	double minImage = Statistic.getMinimum(image);
	double maxImage = Statistic.getMaximum(image);
	
	// Compute the parameter to rescale the gray levels
	double a;
	if (minImage-maxImage == 0) {
		a = 1.0;
		minImage = (maxGoal-minGoal)/2.0;
	}
	else {
		if ( Math.abs(maxImage) > Math.abs(minImage) )
			a = (maxGoal-center) / Math.abs(maxImage);
		else
			a = (center-minGoal) / Math.abs(minImage);
	}
	
	final int nx=image.getXSize();
	final int ny=image.getYSize();
	final int nz=image.getZSize(); 
	
	double row[]=new double[nx];
	image.setDirection(Data.X);
	
	for (int z=0; z<nz; z++)
	  for (int y=0; y<ny; y++) {
		image.get(0, y, z, row);
		for (int x=0; x<nx; x++) {
			row[x] *= a;
			row[x] += center;
		}
		image.put(0, y, z, row);
	  }
				
}

/**
* Apply a rescaling.
*
* Stretches the contrast inside subbands
* so that the gray levels are in the range : the positivy values
* are stretched to center-max and the negative values are stretched 
* to min-center and a contrast stretching is applied on the low band
*
* @param coef			a Data object representing the wavelet coefficients
* @param iterations    	number of iterations
*/
static final public void doRescale3DNonCenter(Data coef, final int[] iterations)
{
	int iterationX=iterations[0];
	int iterationY=iterations[1];
	int iterationZ=iterations[2];

	int nx = coef.getXSize();
	int ny = coef.getYSize();
	int nz = coef.getZSize();
	
	final int type = coef.getType();
	
	coef.setDirection(Data.XYZ);

    final boolean offsets[][] = {
      /* LLL */ { false, false, false },
      /* LLH */ { false, false, true  },
      /* LHL */ { false, true,  false },
      /* LHH */ { false, true,  true  },
      /* HLL */ { true,  false, false },
      /* HLH */ { true,  false, true  },
      /* HHL */ { true,  true,  false },
      /* HHH */ { true,  true,  true  }
    };

	Data sub;
	double subd[][][];
	
	for ( int i=0; i<Math.max(iterationX,Math.max(iterationY,iterationZ)); i++) {
		if (iterationX>i) nx = nx / 2;
		if (iterationY>i) ny = ny / 2;
		if (iterationZ>i) nz = nz / 2;
		
		sub = Data.create(nx, ny, nz, type);
		
		// all subbands except LLL
		for ( int j=1; j<8; j++ ) {
			if ( !( (iterationX<=i && offsets[j][0]) ||
				    (iterationY<=i && offsets[j][1]) ||
				    (iterationZ<=i && offsets[j][2]) ) ) {
				coef.get( (offsets[j][0]) ? nx : 0,
						  (offsets[j][1]) ? ny : 0,
						  (offsets[j][2]) ? nz : 0 , sub); 
				normalizeContrastNonCentered(sub);
				coef.put( (offsets[j][0]) ? nx : 0,
						  (offsets[j][1]) ? ny : 0,
						  (offsets[j][2]) ? nz : 0 , sub);
			}
		}
	}

	// lowest LLL
	sub = Data.create(nx,ny,nz,type);
	coef.get(0, 0, 0, sub);
	sub.normalizeContrast();
	coef.put(0, 0, 0, sub);
}

/**
* Stretches the contrast inside an image so that the gray levels are
* in the range thevalues are stretched to min-max. 
*/
private static void normalizeContrastNonCentered(Data image)
{
	double minGoal = 0.0;
	double maxGoal = 255.0;
	
	// Get the min and max
	double minImage = Statistic.getMinimum(image);
	double maxImage = Statistic.getMaximum(image);
	
	// Compute the parameter to rescale the gray levels
	double a;
	if ( minImage-maxImage == 0) {
		a = 1.0;
		minImage = (maxGoal-minGoal)/2.0;
	}
	else
		a = (maxGoal-minGoal) / (maxImage-minImage);

	final int nx=image.getXSize();
	final int ny=image.getYSize();
	final int nz=image.getZSize(); 
	
	double row[]=new double[nx];
	image.setDirection(Data.X);
	
	for (int z=0; z<nz; z++)
	  for (int y=0; y<ny; y++) {
		image.get(0, y, z, row);
		for (int x=0; x<nx; x++) {
			row[x] = row[x] - minGoal;
		}
		image.put(0, y, z, row);
	  }
				
}

} // end of class